'use strict';

/* --------------------------------------------------------------
 filemanager.js 2020-06-26
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Filemanager Widget
 *
 * Creates an input field and a button in order to make it possible to upload files or to select already
 * uploaded files. This widget checks if the responsive filemanager is present, if it is, the responsive
 * filemanager will be used, else a fallback will be used, which is an input field of type 'file'.
 *
 * ### Options
 *
 * **Type | `data-filemanager-type` | String | Optional**
 *
 * Provide the allowed file upload type. Currently there are 3 options which are:
 *    * 'all' - All file types are allowed
 *    * 'videos' - Only video files are allowed
 *    * 'images' - Only images are allowed
 * If you don't provide any type, the filemanager will default to 'all'.
 *
 * **Content Directory | `data-filemanager-content-directory` | String | Required**
 *
 * Provide the directory which should be opened when the filemanager gets opened, e.g. 'media'.
 * You can also provide a path from the root of your shop e.g 'images/slider_images'.
 *
 * **Name | `data-filemanager-name` | String | Required**
 *
 * The name of the input field. It will be set as the HTML name attribute.
 *
 * **Previous File | `data-filemanager-previous-file` | String | Optional**
 *
 * Name of the previous file. The name will be used in order to auto fill the input field.
 *
 * **Page | `data-filemanager-page` | String | Optional**
 *
 * The name of the current page in snake case, for example: new_category or responsive_filemanager.
 * This option will be used in order to load custom configuration files for the responsive file manager like
 * responsive_filemanager.config.php. These custom configuration files will be available or should be created
 * in the 'page' directory of the responsive file manager.
 *
 * **Page Active | `data-filemanager-page-active` | Boolean | Required**
 *
 * This option is required in order to check whether the file manager module is active, and if the configuration
 * option from the file manager is set to active, for the current page. If the module is not active, or active
 * in general but not active for the current page, the fallback will be used, which is a standard input field.
 *
 * ### Example
 *
 * ```html
 * <div data-gx-widget="filemanager"
 *     data-filemanager-name="categories_icon" // Required
 *     data-filemanager-type="images" // Optional
 *     data-filemanager-content-directory="images/categories/icons" // Required
 *     data-filemanager-previous-file="filename.extension" // Optional
 *     data-filemanager-page="responsive_filemanager" // Optional
 *     data-filemanager-page-active="true"> // Required
 * </div>
 * ```
 *
 * @module Admin/Widgets/filemanager
 */
gx.widgets.module('filemanager', ['xhr', 'modal'],

/** @lends module:Widgets/filemanager */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLE DEFINITION
    // ------------------------------------------------------------------------

    /**
     * Widget Reference
     *
     * @type {object}
     */

    var _this = this;

    var $this = $(this);

    /**
     * Default Widget Options
     *
     * @type {object}
     */
    var defaults = {};

    /**
     * Final Widget Options
     *
     * @type {object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Module Object
     *
     * @type {object}
     */
    var module = {};

    /**
     * Id of the file manager input field.
     *
     * @type {string}
     */
    var fieldId = void 0;

    /**
     * Ajax request url fetch the file managers configuration settings.
     *
     * @type {string}
     */
    var fileManagerConfigurationUrl = jse.core.config.get('appUrl') + '/admin/admin.php?do=ResponsiveFileManagerModuleCenterModule/GetConfiguration';

    /**
     *  Cache key.
     *
     * @type {string}
     */
    var cacheKey = 'responsiveFileManager';

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    // ------------------------------------------------------------------------
    // FILE MANAGER CONFIGURATIONS
    // ------------------------------------------------------------------------
    /**
     * Returns the allowed file type as an integer, which is mapped
     * for the external Responsive Filemanager plugin. It will be used
     * as a GET parameter in the URL for the file manager.
     *
     * @returns {number} Flag integer value between 0 and 3.
     */
    var _getFMType = function _getFMType() {
        switch (options.type) {
            case 'images':
                return 1;
            case 'all':
                return 2;
            case 'videos':
                return 3;
            default:
                return 0;
        }
    };

    /**
     * File managers request url.
     *
     * @returns {string} Request url of file manager.
     */
    var _getFMUrl = function _getFMUrl() {
        // Language parameter used for the file manager
        var lang = jse.core.registry.get('languageId') === 2 ? 'de' : 'en_EN';

        // Don't use the popup mode if the file manager will be opened in a modal.
        var popUp = _isCompatibilityModeEnabled() ? 1 : '';

        return jse.core.config.get('appUrl') + '/' + 'ResponsiveFilemanager/filemanager/filemanager.php?type=' + _getFMType() + _getSubDirectoryQueryString() + '&field_id=' + fieldId + '&popup=' + popUp + '&relative_url=1&lang=' + lang + _getPageQueryString();
    };

    /**
     * Returns the 'sub_folder' query argument for the file manager request.
     *
     * @returns {string} Query parameter for file manager request to set the root directory.
     */
    var _getSubDirectoryQueryString = function _getSubDirectoryQueryString() {
        if (options.contentDirectory !== undefined) {
            if (options.targetDirectory !== undefined) {
                return '&target_path=' + options.targetDirectory + '&sub_folder=' + options.contentDirectory;
            }
            return '&sub_folder=' + options.contentDirectory;
        }

        return '';
    };

    /**
     * Returns the 'page' query string for the file manager request.
     *
     * @returns {string} Query parameter for the file manager request to load a custom configuration file.
     */
    var _getPageQueryString = function _getPageQueryString() {
        if (options.page !== undefined) {
            return '&page=' + options.page;
        }

        return '';
    };

    /**
     * Generates a global unique identifier for each input that is generated by this widget.
     * This ID will be used in order to identify an input fields. With the help of this ID,
     * the widget knows, in which input field the file name of the chose file should be entered.
     *
     * @returns {string} Global unique identifier as string.
     */
    var guidGenerator = function guidGenerator() {
        var s4 = function s4() {
            return ((1 + Math.random()) * 0x10000 | 0).toString(16).substring(1);
        };

        return s4() + s4() + "-" + s4() + "-" + s4() + "-" + s4() + "-" + s4() + s4() + s4();
    };

    // ------------------------------------------------------------------------
    // CREATING THE FILE MANAGER
    // ------------------------------------------------------------------------
    /**
     * Factory, which creates either the responsive file manager or the fallback,
     * which is a standard input field of the type 'file'.
     *
     * @type {{responsive: (function()), fallback: (function())}}
     */
    var factory = {
        responsive: function responsive() {
            var $uploadIcon = $('<i/>', {
                'class': 'fa fa-upload',
                'aria-hidden': true
            });

            var $removeIcon = $('<i/>', {
                'class': 'fa fa-remove',
                'aria-hidden': true
            });

            var $input = $('<input/>', {
                'type': 'text',
                'name': options.name,
                'id': fieldId,
                'class': 'form-control',
                'readonly': 'readonly',
                'on': {
                    'change': function change() {
                        return _getPath();
                    }
                }
            });

            // Auto fill the input field with the previous file name
            if (options.previousFile) {
                $input.val(options.previousFile);
            }

            var $uploadButton = $('<a/>', {
                'class': 'btn responsive-file-manager',
                'type': 'button',
                'html': $uploadIcon,
                'on': {
                    'click': function click() {
                        return _openFileManager();
                    }
                }
            });

            var $removeButton = $('<a/>', {
                'class': 'btn responsive-file-manager',
                'type': 'button',
                'html': $removeIcon,
                'on': {
                    'click': function click() {
                        return $input.val('');
                    }
                }
            });

            var $span = $('<span/>', {
                'class': 'input-group-btn'
            });

            var $container = $('<div/>', {
                'class': 'input-group responsive-file-manager'
            });

            $span.append($uploadButton);
            $span.append($removeButton);

            $container.append($input).append($span);
            $this.append($container);
        },

        fallback: function fallback() {
            var $input = $('<input/>', {
                'name': options.name,
                'type': 'file'
            });

            $this.append($input);
        }
    };

    /**
     /**
     * Creates the widget after the request the responsive file manager
     * request is being made. After the request, either the 'responsive'
     * widget is created or the fallback, depending on if the file manager
     * is available.
     *
     * @param done Done callback function for module.init.
     */
    var _createWidget = function _createWidget(done) {
        jse.libs.xhr.get({ url: fileManagerConfigurationUrl }).done(function (response) {
            return jse.core.registry.set(cacheKey, response.isInstalled ? 'responsive' : 'fallback');
        }).fail(function () {
            return jse.core.registry.set(cacheKey, 'fallback');
        }).always(function () {
            // Create the file manager or fallback.
            factory[jse.core.registry.get(cacheKey)]();
            done();
        });
    };

    /**
     * Creates the widget when the cache key changes from pending.
     * After the cache key changed to either the 'responsive' or 'fallback',
     * the according widget will be created, depending on if the file manager
     * is available.
     *
     * @param done Done callback function for module.init.
     */
    var _createWidgetWhenCacheKeyAvailable = function _createWidgetWhenCacheKeyAvailable(done) {
        var interval = setInterval(function () {
            if (jse.core.registry.get(cacheKey) !== 'pending') {
                clearInterval(interval);

                // Create the file manager or fallback.
                factory[jse.core.registry.get(cacheKey)]();
                done();
            }
        }, 100);
    };

    // ------------------------------------------------------------------------
    // OPENING THE FILE MANAGER
    // ------------------------------------------------------------------------
    /**
     * Opens the file manager in a new window popup.
     */
    var _openFMPopup = function _openFMPopup() {
        var w = 990;
        var h = 600;
        var l = Math.floor((screen.width - w) / 2);
        var t = Math.floor((screen.height - h) / 2);

        window.open(_getFMUrl(), 'ResponsiveFilemanager', "scrollbars=1,width=" + w + ",height=" + h + ",top=" + t + ",left=" + l);
    };

    /**
     * Opens the file manager in a bootstrap modal.
     */
    var _openFMModal = function _openFMModal() {

        // Use the fallback if bootstraps modal function is not available.
        if ($.fn.modal === undefined) {
            return _openFMPopup();
        }

        var iFrame = '<iframe id="filemanager" src="' + _getFMUrl() + '" width="100%" height="550" frameborder="0"\n\t\t\t\t\t\t\t\t\t\t\t\t\t\t\tclass="responsive-filemanager"></iframe>';

        jse.libs.modal.showMessage('Filemanager', iFrame);
        _makeModalLarge();
    };

    /**
     * Makes the modal large by adding the modal-lg css class.
     */
    var _makeModalLarge = function _makeModalLarge() {
        $('.modal-dialog:last').addClass('modal-lg');
    };

    /**
     * Checks if compatibility mode is active.
     *
     * @returns {boolean} True on compatibility mode, false otherwise.
     */
    var _isCompatibilityModeEnabled = function _isCompatibilityModeEnabled() {
        return $('body.gx-compatibility').length !== 0;
    };

    /**
     * Opens the file manager in a modal, dialog or window with the priority in
     * the same order. If bootstrap is not available, the file
     * manager will be opened in a new window.
     */
    var _openFileManager = function _openFileManager() {
        if (_isCompatibilityModeEnabled()) {
            return _openFMPopup();
        }

        _openFMModal();
    };

    var _getPath = function _getPath() {
        var input = $(_this).find('input');
        input.val(_removePath(input.val()));
    };

    var _removePath = function _removePath(path) {
        if (options.removePath === undefined) {
            if (options.contentDirectory === '/') {
                return path;
            } else {
                if (options.contentDirectory.endsWith('/')) {
                    return path.replace(options.contentDirectory, '');
                } else {
                    return path.replace(options.contentDirectory + '/', '');
                }
            }
        }
        return path.replace(options.removePath, '');
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    /**
     * Initialize method of the widget, called by the engine.
     */
    module.init = function (done) {
        fieldId = guidGenerator();

        // Required option not provided
        if (options.contentDirectory === undefined || options.contentDirectory === '') {
            jse.core.debug.error('content-directory attribute was not provided for the "filemanager" widget.');
            return;
        }

        // Required option not provided
        if (options.name === undefined || options.name === '') {
            jse.core.debug.error('name attribute was not provided for the "filemanager" widget.');
            return;
        }

        // Required option not provided
        if (options.pageActive === undefined) {
            jse.core.debug.error('page-active attribute was not provided for the "filemanager" widget.');
            return;
        }

        // Module is not active at all or not active for the used page.
        if (!options.pageActive) {
            factory.fallback();
            done();
            return;
        }

        // No cache key available yet. Create the widget and set the cache key to 'fallback' or 'responsive'
        // after the responsive has arrived (done by the _createWidget function).
        if (jse.core.registry.get(cacheKey) === undefined) {
            jse.core.registry.set(cacheKey, 'pending');
            _createWidget(done);
            return;
        }

        // Cache key is on 'pending' which means we have to wait until the key changes (done by the _createWidget function).
        // Afterwards we can create the correct widget.
        if (jse.core.registry.get(cacheKey) === 'pending') {
            _createWidgetWhenCacheKeyAvailable(done);
            return;
        }

        // Build the fallback or responsive file manager.
        factory[jse.core.registry.get(cacheKey)]();
    };

    // Return data to module engine.
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
